import React, { useEffect, useState } from "react";
import { useParams, Link } from "react-router-dom";
import { api } from "../config/api";
import { Loader, ArrowLeft, Download } from "lucide-react";

export const FormResponses = () => {
  const { id } = useParams();
  const [form, setForm] = useState(null);
  const [responses, setResponses] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchData = async () => {
      try {
        const [formRes, responsesRes] = await Promise.all([
          api.get(`/api/forms/${id}`),
          api.get(`/api/responses?formId=${id}`),
        ]);
        setForm(formRes.data);
        setResponses(responsesRes.data);
      } catch (err) {
        console.error("Failed to fetch data", err);
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, [id]);

  const getHeaders = () => {
    if (!form) return [];
    const headers = [];
    form.steps.forEach((step) => {
      step.fields.forEach((field) => {
        headers.push(field.label);
      });
    });
    return headers;
  };

  const downloadCSV = () => {
    if (!responses.length) return;

    const headers = getHeaders();
    const csvRows = [];

    // Add headers
    csvRows.push(["Submitted At", ...headers].join(","));

    // Add data
    responses.forEach((response) => {
      const row = [
        new Date(response.submittedAt).toLocaleString(),
        ...headers.map((header) => {
          const answer = response.answers[header] || "";
          return `"${answer.replace(/"/g, '""')}"`; // Escape quotes
        }),
      ];
      csvRows.push(row.join(","));
    });

    const blob = new Blob([csvRows.join("\n")], { type: "text/csv" });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `responses-${form.title || "form"}.csv`;
    a.click();
    window.URL.revokeObjectURL(url);
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <Loader className="animate-spin text-blue-600" size={32} />
      </div>
    );
  }

  if (!form) return <div className="text-center p-8">Form not found</div>;

  const headers = getHeaders();

  return (
    <div className="min-h-screen bg-gray-50 p-8 font-sans text-neutral-900">
      <div className="max-w-7xl mx-auto">
        <div className="mb-8">
          <Link
            to="/"
            className="text-blue-600 hover:text-blue-800 flex items-center gap-2 mb-4 font-medium"
          >
            <ArrowLeft size={20} />
            Back to Forms
          </Link>
          <div className="flex justify-between items-end">
            <div>
              <h1 className="text-3xl font-light text-neutral-800 mb-2">
                {form.title}
              </h1>
              <p className="text-neutral-500">{responses.length} responses</p>
            </div>
            {responses.length > 0 && (
              <button
                onClick={downloadCSV}
                className="flex items-center gap-2 bg-white border border-gray-200 px-4 py-2 rounded-lg text-gray-700 hover:bg-gray-50 hover:border-blue-200 hover:text-blue-600 transition-colors shadow-sm"
              >
                <Download size={18} />
                Export CSV
              </button>
            )}
          </div>
        </div>

        {responses.length === 0 ? (
          <div className="bg-white p-12 rounded-xl shadow-sm border border-gray-100 text-center">
            <p className="text-gray-500 text-lg">No responses yet.</p>
            <p className="text-gray-400 mt-2">
              Share your form to start collecting data.
            </p>
          </div>
        ) : (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden overflow-x-auto">
            <table className="w-full text-left border-collapse">
              <thead>
                <tr className="bg-gray-50 border-b border-gray-200">
                  <th className="p-4 font-medium text-gray-500 text-sm uppercase tracking-wider whitespace-nowrap">
                    Submitted At
                  </th>
                  {headers.map((header, i) => (
                    <th
                      key={i}
                      className="p-4 font-medium text-gray-500 text-sm uppercase tracking-wider whitespace-nowrap min-w-[200px]"
                    >
                      {header}
                    </th>
                  ))}
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-100">
                {responses.map((response) => (
                  <tr
                    key={response._id}
                    className="hover:bg-blue-50/30 transition-colors"
                  >
                    <td className="p-4 text-gray-600 whitespace-nowrap text-sm">
                      {new Date(response.submittedAt).toLocaleString()}
                    </td>
                    {headers.map((header, i) => (
                      <td key={i} className="p-4 text-gray-800">
                        {response.answers[header] ? (
                          <span
                            className="line-clamp-2"
                            title={response.answers[header]}
                          >
                            {response.answers[header]}
                          </span>
                        ) : (
                          <span className="text-gray-300 italic">Empty</span>
                        )}
                      </td>
                    ))}
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  );
};
